﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using BMS.Facade.Data;
using BMS.ServicesWrapper.BMService;
using BMS.Facade.Translators;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using InfoWorld.EVS.CTSMAPI;
using BMS.ServicesWrapper.EVS;
using BMS.ServicesWrapper.WF;
using BMS.VistaWorker2.Writer.Exceptions;
using BMS.DataContracts;


namespace BMS.VistaWorker2.Writer.Implementation.Concrete.EIS
{
    /// <summary>
    /// The room bed writer.
    /// </summary>
    class RoomBedWriter : BaseEisWriter<RoomBed, Bed>
    {        
        private bool entityHasChanged;
        private bool bedUnavailableHasChanged;
        private List<Ward> entityWards;

        private bool EmtpyWards(RoomBed bed)
        {
            if (bed.WardsWhichCanAssign != null && bed.WardsWhichCanAssign.Count > 0)
                return false;
            return true;
        }

        protected override Bed GetBmsEntity(RoomBed entity)
        {
            Bed  bed = base.GetBmsEntity(entity);
            if (EmtpyWards(entity) && bed == null)
                throw new WriterException(entity, string.Format("Bed {0} has no wards", entity.Name));
            entityWards = GetWardsWhitchCanAssign(entity);            
            return bed;
        }

        private List<Ward> GetWardsWhitchCanAssign(RoomBed entity)
        {
            if (entity.WardsWhichCanAssign == null)
                return new List<Ward>();
            List<Ward> result = entity.Wards.Select(ward => InsertIfNullOrUpdateIfDirty<WardLocation, Ward>(ward)).ToList();
            return result;
        }

        public override void InitCache()
        {
            Data.SetCacheForWard();
            Data.SetCacheForBed();
            Data.SetCacheForHospitalLocation();
            Data.SetCacheForSpecialty();
        }

        /// <summary>
        /// Adds the modifications for BMS entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void TranslateFromVistaToBMS(RoomBed entity, Bed bmsEntity)
        {
            bmsEntity.WardList = entityWards;
            bmsEntity.Ien = entity.IEN;
            bmsEntity.Name = entity.Name;
            bmsEntity.IsBedHold = GetIsBedHold(entity);
            bmsEntity.VistaSite = VistaSite;
        }

        private static bool GetIsBedHold(RoomBed entity)
        {
            return !(string.IsNullOrEmpty(entity.BedHoldPatientIen)) && entity.BedHoldPatientIen != "0";
        }

        private bool EqualWardsWhichCanAssign(RoomBed entity, IList<Ward> wards)
        {
            bool emptyEntityWards = entity.WardsWhichCanAssign == null || entity.WardsWhichCanAssign.Count == 0;
            bool emptyBmsEntityWards = wards == null || wards.Count == 0;
            if (emptyBmsEntityWards ^ emptyEntityWards)
                return false;
            if (emptyEntityWards)
                return true;
            if (entityWards.Count != wards.Count)
                return false;
            entityWards = entityWards.OrderBy(a => a.Id.extension).ToList();
            wards = wards.OrderBy(a => a.Id.extension).ToList();
            for (int i = 0; i < entityWards.Count; i++)
            {
                if (string.Compare(entityWards[i].Id.extension, wards[i].Id.extension, true) != 0)
                    return false;
            }
            return true;
        }        

        private bool GetEntityHasChanged(RoomBed entity, Bed bmsEntity)
        {
            return entity.Name != bmsEntity.Name ||
                     GetIsBedHold(entity) != bmsEntity.IsBedHold ||
                     !EqualWardsWhichCanAssign(entity, bmsEntity.WardList);            
        }

        protected override bool AreFieldsEqual(RoomBed entity, Bed bmsEntity)
        {
            BedUnavailableWriter bedUnavailableWriter = new BedUnavailableWriter();
            entityHasChanged = GetEntityHasChanged(entity, bmsEntity);
            bedUnavailableHasChanged = bedUnavailableWriter.GetBedUnavailableHasChanged(Data, entity, bmsEntity);
            return !(entityHasChanged || bedUnavailableHasChanged);
        }

        protected override void InsertBmsEntity(RoomBed vistaEntity, Bed bmsEntity)
        {
            base.InsertBmsEntity(vistaEntity, bmsEntity);
            BedUnavailableWriter bedUnavailableWriter = new BedUnavailableWriter();
            bedUnavailableWriter.ChangeBedUnavailable(Data, vistaEntity, bmsEntity);
        }

        protected override void UpdateBmsEntity(RoomBed vistaEntity, Bed bmsEntity)
        {
            if (entityHasChanged)
            {
                base.UpdateBmsEntity(vistaEntity, bmsEntity);
            }
            if (bedUnavailableHasChanged)
            {
                BedUnavailableWriter bedUnavailableWriter = new BedUnavailableWriter();
                bedUnavailableWriter.ChangeBedUnavailable(Data, vistaEntity, bmsEntity);
            }
        }

        protected override IEntityDal<Bed> Dal
        {
            get { return Data.BedDal; }
        }
    }
}
